#!/usr/bin/env bash
set -euo pipefail

# =========================
# Config (overridable via env vars)
# =========================
BASE_URL="${BASE_URL:-http://<BASE_URL>}"
API_KEY="${API_KEY:-<API_KEY>}"
MODEL_NAME="${MODEL_NAME:-<MODEL_NAME>}"
EMB_MODEL_NAME="${EMB_MODEL_NAME:-<EMB_MODEL_NAME>}"
IMAGE_URL="${IMAGE_URL:-}"

# Optional: timeouts
CURL_CONNECT_TIMEOUT="${CURL_CONNECT_TIMEOUT:-5}"
CURL_MAX_TIME="${CURL_MAX_TIME:-60}"

# =========================
# Tool checks
# =========================
need_cmd() {
  command -v "$1" >/dev/null 2>&1 || {
    echo "[ERROR] Missing dependency: $1"
    exit 2
  }
}

need_cmd curl
need_cmd jq

# Global stats
FAILED_COUNT=0
declare -a FAILED_TITLES=()
declare -a FAILED_DETAILS=()

# =========================
# Common POST helper: returns body + http_code
# =========================
http_post_json() {
  local url="$1"
  local json="$2"
  local err_file
  local resp
  local curl_code

  # Output format: body\nHTTP_CODE
  err_file="$(mktemp)"
  set +e
  resp="$(curl -sS \
    --connect-timeout "$CURL_CONNECT_TIMEOUT" \
    --max-time "$CURL_MAX_TIME" \
    -H "Content-Type: application/json" \
    -H "Authorization: Bearer ${API_KEY}" \
    -d "$json" \
    -w "\n%{http_code}" \
    "$url" 2>"$err_file")"
  curl_code=$?
  set -e

  if [[ $curl_code -ne 0 ]]; then
    local err_msg
    err_msg="$(cat "$err_file" || true)"
    rm -f "$err_file"
    case "$curl_code" in
      6) fail "Network error: DNS resolve failed" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
      7) fail "Network error: connection failed (port closed/service down)" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
      28) fail "Network error: request timed out" "curl_exit=$curl_code\nconnect_timeout=${CURL_CONNECT_TIMEOUT}s max_time=${CURL_MAX_TIME}s\nURL=$url\nDetail:\n$err_msg" ;;
      35) fail "Network error: TLS/SSL handshake failed" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
      52) fail "Network error: empty reply from server" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
      56) fail "Network error: connection broken (recv failed)" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
      *) fail "Network error: curl failed" "curl_exit=$curl_code\nURL=$url\nDetail:\n$err_msg" ;;
    esac
    return 1
  fi

  rm -f "$err_file"
  printf '%s' "$resp"
}

fail() {
  local title="$1"
  local detail="$2"
  FAILED_COUNT=$((FAILED_COUNT + 1))
  FAILED_TITLES+=("$title")
  FAILED_DETAILS+=("$detail")
  echo
  echo "===================="
  echo "[ERROR] $title"
  echo "--------------------"
  printf '%b\n' "$detail"
  echo "===================="
  echo
}

pass() {
  echo "[OK] $1"
}

# =========================
# 1) Chat Completions check
# Expect: HTTP 200; choices[0].message.content not empty; object=chat.completion (if returned)
# =========================
check_chat() {
  local url="${BASE_URL%/}/chat/completions"
  local payload
  payload="$(jq -nc --arg m "$MODEL_NAME" '{
    model: $m,
    messages: [{role:"user", content:"Hello"}],
    stream: false
  }')"

  local resp body code
  if ! resp="$(http_post_json "$url" "$payload")"; then
    return 1
  fi
  body="$(printf '%s' "$resp" | sed '$d')"
  code="$(printf '%s' "$resp" | tail -n 1)"

  if [[ "$code" != "200" ]]; then
    fail "Chat endpoint HTTP != 200" "HTTP=$code\nURL=$url\nResponse:\n$body"
    return 1
  fi

  # If error field exists, fail immediately
  if jq -e '.error? // empty' >/dev/null 2>&1 <<<"$body"; then
    fail "Chat returned error" "$(jq -c '.error' <<<"$body")"
    return 1
  fi

  local content
  content="$(jq -r '.choices[0].message.content // empty' <<<"$body")"
  if [[ -z "$content" ]]; then
    fail "Chat content is empty or response structure is unexpected" "URL=$url\nResponse:\n$body"
    return 1
  fi

  pass "Chat Completions OK (content is not empty)"
  return 0
}

# =========================
# 2) Embeddings check
# Expect: HTTP 200; data[0].embedding is an array with length > 0
# =========================
check_embeddings() {
  local url="${BASE_URL%/}/embeddings"
  local payload
  payload="$(jq -nc --arg m "$EMB_MODEL_NAME" '{
    model: $m,
    input: "test"
  }')"

  local resp body code
  if ! resp="$(http_post_json "$url" "$payload")"; then
    return 1
  fi
  body="$(printf '%s' "$resp" | sed '$d')"
  code="$(printf '%s' "$resp" | tail -n 1)"

  if [[ "$code" != "200" ]]; then
    fail "Embeddings endpoint HTTP != 200" "HTTP=$code\nURL=$url\nResponse:\n$body"
    return 1
  fi

  if jq -e '.error? // empty' >/dev/null 2>&1 <<<"$body"; then
    fail "Embeddings returned error" "$(jq -c '.error' <<<"$body")"
    return 1
  fi

  local len
  len="$(jq -r '(.data[0].embedding | type) as $t | if $t=="array" then (.data[0].embedding|length) else -1 end' <<<"$body")"
  if [[ ! "$len" =~ ^[0-9]+$ ]]; then
    fail "Embeddings parse failed" "URL=$url\nResponse:\n$body"
    return 1
  fi
  if (( len <= 0 )); then
    fail "Embeddings vector is empty or structure is unexpected" "embedding_length=$len\nURL=$url\nResponse:\n$body"
    return 1
  fi

  pass "Embeddings OK (embedding_length=${len})"
  return 0
}

# =========================
# 3) Tools / Function Call check (minimal pass-through)
# Expect: HTTP 200; choices[0].message.tool_calls is array with length > 0
#         OR finish_reason == "tool_calls"
# Note: Different model implementations may vary; this is a compatible check.
# =========================
check_tools() {
  local url="${BASE_URL%/}/chat/completions"
  local payload
  payload="$(jq -nc --arg m "$MODEL_NAME" '{
    model: $m,
    stream: false,
    messages: [
      {role:"system", content:"You can call tools when needed."},
      {role:"user", content:"Call get_time tool."}
    ],
    tools: [
      {
        type: "function",
        function: {
          name: "get_time",
          description: "Get current time in ISO format",
          parameters: {type:"object", properties:{}}
        }
      }
    ]
  }')"

  local resp body code
  if ! resp="$(http_post_json "$url" "$payload")"; then
    return 1
  fi
  body="$(printf '%s' "$resp" | sed '$d')"
  code="$(printf '%s' "$resp" | tail -n 1)"

  if [[ "$code" != "200" ]]; then
    fail "Tools test HTTP != 200" "HTTP=$code\nURL=$url\nResponse:\n$body"
    return 1
  fi

  if jq -e '.error? // empty' >/dev/null 2>&1 <<<"$body"; then
    fail "Tools test returned error" "$(jq -c '.error' <<<"$body")"
    return 1
  fi

  # Compatible: tool_calls array length > 0 OR finish_reason == tool_calls
  local tc_len finish
  tc_len="$(jq -r '(.choices[0].message.tool_calls // []) | if type=="array" then length else 0 end' <<<"$body")"
  finish="$(jq -r '.choices[0].finish_reason // empty' <<<"$body")"

  if ! { [[ "$tc_len" =~ ^[0-9]+$ ]] && (( tc_len > 0 )); } && [[ "$finish" != "tool_calls" ]]; then
    fail "Tools check failed (no tool_calls)" "tool_calls_length=$tc_len finish_reason=$finish\nURL=$url\nResponse:\n$body"
    return 1
  fi

  pass "Tools OK (tool_calls_length=${tc_len} finish_reason=${finish})"
  return 0
}

# =========================
# 4) Image request check (optional)
# Only runs when IMAGE_URL is set
# =========================
check_image() {
  if [[ -z "$IMAGE_URL" ]]; then
    echo "[INFO] Skip image check (IMAGE_URL not set)"
    return 0
  fi

  local url="${BASE_URL%/}/chat/completions"
  local payload
  payload="$(jq -nc --arg m "$MODEL_NAME" --arg image_url "$IMAGE_URL" '{
    model: $m,
    stream: false,
    messages: [
      {
        role: "user",
        content: [
          {type: "text", text: "Describe the main content of this image in 20 words or less."},
          {type: "image_url", image_url: {url: $image_url}}
        ]
      }
    ]
  }')"

  local resp body code
  if ! resp="$(http_post_json "$url" "$payload")"; then
    return 1
  fi
  body="$(printf '%s' "$resp" | sed '$d')"
  code="$(printf '%s' "$resp" | tail -n 1)"

  if [[ "$code" != "200" ]]; then
    fail "Image request HTTP != 200" "HTTP=$code\nURL=$url\nIMAGE_URL=$IMAGE_URL\nResponse:\n$body"
    return 1
  fi

  if jq -e '.error? // empty' >/dev/null 2>&1 <<<"$body"; then
    fail "Image request returned error (image may be inaccessible or model lacks multimodal)" "$(jq -c '.error' <<<"$body")"
    return 1
  fi

  local content
  content="$(jq -r '.choices[0].message.content // empty' <<<"$body")"
  if [[ -z "$content" ]]; then
    fail "Image response content is empty or structure is unexpected" "URL=$url\nIMAGE_URL=$IMAGE_URL\nResponse:\n$body"
    return 1
  fi

  pass "Image request OK (model can access image)"
  return 0
}

# =========================
# Main flow
# =========================
echo "BASE_URL=$BASE_URL"
echo "MODEL_NAME=$MODEL_NAME"
echo "EMB_MODEL_NAME=$EMB_MODEL_NAME"
if [[ -n "$IMAGE_URL" ]]; then
  echo "IMAGE_URL=$IMAGE_URL"
fi
echo

check_chat || true
check_embeddings || true
check_tools || true
check_image || true

echo
if (( FAILED_COUNT > 0 )); then
  echo "[WARNING] Completed with ${FAILED_COUNT} failure(s) (continued to run remaining checks)"
  echo "Failure details:"
  for i in "${!FAILED_TITLES[@]}"; do
    printf '%d) %s\n' "$((i + 1))" "${FAILED_TITLES[$i]}"
    printf '%b\n' "${FAILED_DETAILS[$i]}"
    echo
  done
  exit 1
fi

echo "All checks passed"
